% Example 2
%
% Worked out example of the workflow to explain the approach of timing 
% recovery on pixel-by-pixel basis in sequential spatial data sets as 
% displayed in Figure 4 of van Belzen et al 2017
%
% Reference:
% van Belzen, J, van de Koppel, J, van der Wal, D, Herman, P.M.J, Dakos, V,
% Kefi, S, Scheffer, M, Bouma, T.J "Timing recovery of ecosystems in 
% sequential remotely sensed and simulated data" Protocol Exchange (2017) 
% DOI: 10.1038/protex.2017.xxx
%
% Version 1.0 (25/07/2014)
% Version 1.1 (25/01/2017)
% Version 1.2 (30/03/2017)
% Coded and developed by Jim van Belzen
% published under the Creative Commons Attribution Non-Commercial license 
% which allows users to read, copy, distribute and make derivative works 
% for noncommercial purposes from the material, as long as the author of 
% the original work is cited.
% This code comes with no warranties or support
% http://people.zeelandnet.nl/jbelzen/


%% Clear workspace and command  window
clear all
clc

on=1;
off=0;


%% Load & prepare example data
%- [ 1.4) Stack grids: ]
Seq = [];
for i = 0:10:500,
    
    % Load snapshots (grids)
    grid = dlmread(['example_data/data_2/t',num2str(i),'.grid']);

    % Stack snapshot (concatenate grids in layers)
    Seq = cat(3, Seq, grid);
    
end
[row, col, lay] = size(Seq);

% Create vector with time stamps of each snapshot
time = linspace(0, col, lay);

% Stress gradient (increas in stress along gradient from left to right)
gr0 = repmat(linspace(0, 1, col),row,1);
gr = gr0;
for i = 1:lay-2,
    gr = cat(3, gr, gr0);
end
    

%% Compute recovery time (Tr) from sequential data
%- [2) Timing recovery in sequential spatial data:]
[Tr, Tr_all, Cens_all, Dist_all] = fTimeRecov(Seq, time);

 
%% Analysis recovery along gradient using binned approach
%- [3) Recovery rates along a stress gradient:]
alpha=0.05;     % stat. significance level used in mle for recovery rate

binmin=0;       % min. range of bins
binmax=1;       % max. range of bins
binstep=1/20;   % bin size

%- Prepare data for binnen
data2bin = Tr_all(:);
bindepend = gr(:);
cens = Cens_all(:);

Cov = Seq(:,:,2:end);
Cov = Cov(:);
Bin = gr(:);

% remove zero's
bindepend = bindepend(data2bin>0);
cens = cens(data2bin>0);
data2bin = data2bin(data2bin>0);

bins = [binmin:binstep:binmax];
n = numel(bins);
datamax = max(data2bin);

stats = [];
for i=1:n-2,
    
    data = data2bin(bindepend>=bins(i) & bindepend<bins(i+1));
    c = cens(bindepend>=bins(i) & bindepend<bins(i+1));
    cov = Cov(Bin>=bins(i) & Bin<bins(i+1));
    
    stats(i,1) = (bins(i) + bins(i+1))/2; % bin centre
    stats(i,2) = numel(data); % number of observations (datapoints)
    stats(i,3) = mean(cov);   % average coverage
    stats(i,4) = mean(data);  % average recovery time (Tr)
    stats(i,5) = std(data);   % standard deviation Tr
    stats(i,6) = std(data)/sqrt(numel(data)); % standard error of Tr
    
    %- biased, no correction for censored data
    [muhat,muci] = expfit(data,alpha);
    stats(i,7) = 1/muhat;     % recovery rate (without censoring correction)
    stats(i,8:9) = 1./muci';  % confidence intervals
    
    %- unbiased, correct for censored data
    [muhat,muci] = expfit(data,alpha,c);
    stats(i,10) = 1/muhat;    % recovery rate (with censoring correction)
    stats(i,11:12) = 1./muci';% confidence intervals
    stats(i,13) = sum(c)/numel(c); % fraction of censored data
    
    %- remove censored data
    data = data(c == 0);
    [muhat,muci] = expfit(data,alpha);
    stats(i,14) = 1/muhat;    % recovery rate (with censoring correction)
    stats(i,15:16) = 1./muci';% confidence interval 
    stats(i,17) = numel(data);% number of datapoints/observations
    
end


%% Graphical output
if on,
    
    % Get screen dimensions
    ScrS = get(0,'ScreenSize'); 
    ScrDim = ScrS(3:4);
    WinDim = floor(ScrDim.*[0.9 0.3]);
    
    % Tr color map
    Trmax = max(Tr(:));
    cmap = [0.75, 0.75, 0.75; 1 1 1;];
    cmap(3:Trmax+2,2) = linspace(1, 0.1, Trmax )';
    cmap(3:Trmax+2,3) = 0.35;
    
    %%- Sequence of snapshots
    WinDim = floor(ScrDim.*[0.3 0.8]);
    Figure1 = figure('Position',[(ScrDim-WinDim)*0.75 WinDim],'color','w');

    subplot(4,1,1)
    colormap(cmap)
    imagesc(Tr)
    axis image
    colorbar

    subplot(4,1,2) 
    plot(stats(:,1),stats(:,3),'.-')
    xlabel('Stress gradient')
    ylabel('Coverage (%)')
    
    subplot(4,1,3) 
    plot(stats(:,1),stats(:,4),'.-')
    xlabel('Stress gradient')
    ylabel('Recovery time, Tr (yr)')
    
    subplot(4,1,4) 
    plot(stats(:,1),stats(:,7),'.-',...
        stats(:,1),stats(:,10),'.-',...
        stats(:,1),stats(:,14),'.-')
    xlabel('Stress gradient')
    ylabel('Recovery rate, \lambda (yr^-^1)')
    legend('biased','unbiased','censored removed') 

end